<?php

namespace App\Http\Controllers;

use App\Models\Producto;
use App\Models\TipoProducto;
use App\Models\Departamento;
use App\Models\ProvieneProducto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\QueryException;
class ProductoController extends Controller
{
    // Mostrar todos los productos
    public function index(Request $request)
    {
        // Obtener los tipos de productos para el filtro
        $tiposProductos = TipoProducto::all();

        // Filtrar productos por los parámetros que el usuario ingrese
        $productos = Producto::with(['tipoProducto', 'departamento', 'provieneProducto']);

        // Filtrar por id_producto si se proporciona
        if ($request->filled('id_producto')) {
            $productos->where('id_producto', $request->id_producto);
        }

        // Filtrar por id_tipoproducto si se proporciona
        if ($request->filled('id_tipoproducto')) {
            $productos->where('id_tipoproducto', $request->id_tipoproducto);
        }

        // Filtrar por descripcion si se proporciona
        if ($request->filled('descripcion')) {
            $productos->where('descripcion', 'like', '%' . $request->descripcion . '%');
        }

        // Paginación de los productos
        $productos = $productos->paginate(10);

        // Pasar los productos y los tipos de productos a la vista
        return view('productos.index', compact('productos', 'tiposProductos'));
    }




    // Mostrar formulario para crear un nuevo producto
    public function create()
    {
        $tiposProductos = TipoProducto::all();
        $departamentos = Departamento::all();
        $provieneProductos = ProvieneProducto::all();
        log::alert("aquui");

        return view('productos.create', compact('tiposProductos', 'departamentos', 'provieneProductos'));
    }

    // Guardar un nuevo producto
    public function store(Request $request)
    {
        Log::info("Intentando crear un nuevo producto", ['data' => $request->all()]);

        try {
            // Validación
            $validatedData = $request->validate([
                'descripcion' => 'required|string|max:255',
                'cantidad' => 'required|integer',
                'fecha_caducidad' => 'nullable|date',
                'mantenimiento' => 'nullable|string',
                'id_tipoproducto' => 'required|exists:tipo_productos,id_tipo',
                'id_departamento' => 'required|exists:departamentos,id_departamento',
                'id_proviene_producto' => 'required|exists:proviene_productos,id_proviene_producto',
                'estado' => 'required|boolean',
                'imagen' => 'nullable|image|max:2048', // Asegurar validación de imagen
            ]);

            DB::beginTransaction();

            // Si hay imagen, convertirla a base64
            if ($request->hasFile('imagen')) {
                // Obtener el contenido de la imagen
                $imageContent = file_get_contents($request->file('imagen')->getRealPath());
                // Convertir la imagen a base64
                $base64Image = base64_encode($imageContent);
                // Guardar la imagen en base64 en el arreglo de datos
                $validatedData['imagen'] = $base64Image;
            }

            // Crear el producto
            $producto = Producto::create($validatedData);

            DB::commit();

            Log::info("Producto creado correctamente", ['producto_id' => $producto->id]);

            return redirect()->route('productos.index')->with('success', 'Producto creado correctamente');

        } catch (QueryException $e) {
            DB::rollBack();
            Log::error("Error en la consulta SQL al crear producto", [
                'error' => $e->getMessage(),
                'code' => $e->getCode()
            ]);

            return back()->with('error', 'Ocurrió un error al guardar el producto. Verifica los datos.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::critical("Error inesperado al crear producto", [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);

            return back()->with('error', 'Error inesperado. Contacte con el administrador.');
        }
    }



    // Mostrar formulario para editar un producto
    public function edit($id)
    {
        $producto = Producto::findOrFail($id);
        $tiposProductos = TipoProducto::all();
        $departamentos = Departamento::all();
        $provieneProductos = ProvieneProducto::all();

        return view('productos.edit', compact('producto', 'tiposProductos', 'departamentos', 'provieneProductos'));
    }

    // Actualizar un producto
    public function update(Request $request, $id)
    {
        $request->validate([
            'descripcion' => 'required|string|max:255',
            'cantidad' => 'required|integer',
            'fecha_caducidad' => 'nullable|date',
            'mantenimiento' => 'nullable|string',
            'id_tipoproducto' => 'required|exists:tipo_productos,id_tipo',
            'id_departamento' => 'required|exists:departamentos,id_departamento',
            'id_proviene_producto' => 'required|exists:proviene_productos,id_proviene_producto',
            'estado' => 'required|boolean',
            'imagen' => 'nullable|image|max:2048', // Validación para la imagen
        ]);

        $producto = Producto::findOrFail($id);

        // Preparar los datos a actualizar
        $data = $request->all();

        // Si se sube una nueva imagen, convertirla a base64
        if ($request->hasFile('imagen')) {
            $imagePath = $request->file('imagen')->getRealPath();
            $imageData = file_get_contents($imagePath);
            $data['imagen'] = base64_encode($imageData);
        } else {
            // Si no hay nueva imagen, mantenemos la imagen existente
            $data['imagen'] = $producto->imagen;
        }

        // Actualizar el producto
        $producto->update($data);

        return redirect()->route('productos.index')->with('success', 'Producto actualizado correctamente');
    }


    // Eliminar un producto
    public function destroy($id)
    {
        $producto = Producto::findOrFail($id);
        $producto->delete();

        return redirect()->route('productos.index')->with('success', 'Producto eliminado correctamente');
    }
    public function showWithImageFromWelcome(Request $request)
{
    // Log para saber si el usuario está viendo los productos
    $departamentos = Departamento::all();
    $tiposProductos = TipoProducto::all();
    $provieneProductos = ProvieneProducto::all();

    // Realizar la consulta de productos con filtros
    $productos = Producto::with(['tipoProducto', 'departamento', 'provieneProducto'])
        ->when($request->filled('id_producto'), function ($query) use ($request) {
            $query->where('id_producto', $request->id_producto);
        })
        ->when($request->filled('id_tipoproducto'), function ($query) use ($request) {
            $query->where('id_tipoproducto', $request->id_tipoproducto);
        })
        ->when($request->filled('descripcion'), function ($query) use ($request) {
            $query->where('descripcion', 'like', '%' . $request->descripcion . '%');
        })
        ->when($request->filled('departamento'), function ($query) use ($request) {
            $query->where('id_departamento', $request->departamento);
        })
        ->when($request->filled('proviene_producto'), function ($query) use ($request) {
            $query->where('id_proviene_producto', $request->proviene_producto);
        })
        ->paginate(10);

    // Convertir la imagen a base64 si no está en ese formato
    foreach ($productos as $producto) {
        if ($producto->imagen) {
            // Si la imagen no está en formato base64, la convierto
            if (!str_starts_with($producto->imagen, 'data:image')) {
                $producto->imagen = 'data:image/jpeg;base64,' . $producto->imagen;
            }
        }
    }

    // Devolver la vista con los productos y tipos de productos
    return view('welcome', compact('productos', 'tiposProductos', 'provieneProductos', 'departamentos'));}




}

